#!/usr/bin/env python3
# Message Authenticator v0.92b_022525. Visit https://kf7mix.com/ for information
# Special thanks to KD0QYN, KN4AM, and everyone else who contributed (see changelog)
#
# MIT License, Copyright 2025 Joseph D Lyman KF7MIX --- Permission is hereby granted,  free of charge, to any person obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:  The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.  The Software IS PROVIDED "AS IS",  WITHOUT WARRANTY OF ANY KIND,  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES  OF  MERCHANTABILITY,  FITNESS OR A PARTICULAR PURPOSE AND  NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR  COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,  DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import tkinter as tk
from tkinter import *
from tkinter import ttk, messagebox
import tkinter.font as tkFont
import hashlib
import time
import datetime
import re
import random
import string

swname = "Message Authenticator"
fromtext = "de KF7MIX"
swversion = "0.92b"

progsize = "500x620"
settings = []

class App(tk.Tk):
    def __init__(self):
        super().__init__()

        self.title(swname+" "+fromtext+" (v"+swversion+")")
        self.geometry(progsize)
        self.minsize(500,620)

        # Init the gui
        # Menu
        self.menubar = Menu(self)
        self.toolsmenu = Menu(self.menubar, tearoff = 0)
        self.toolsmenu.add_command(label = 'Generate 24 Keys ^25', command=lambda qty=24: self.codekey_gen(qty,25,0))
        self.toolsmenu.add_command(label = 'Generate 24 Keys ^36', command=lambda qty=24: self.codekey_gen(qty,36,0))
        self.toolsmenu.add_command(label = 'Generate 12 Keys ^28 Ordered', command=lambda qty=12: self.codekey_gen(qty,28,1))
        self.toolsmenu.add_command(label = 'Generate 12 Keys ^40 Ordered', command=lambda qty=12: self.codekey_gen(qty,40,1))
        self.toolsmenu.add_command(label = 'Generate 31 Keys ^28 Ordered', command=lambda qty=31: self.codekey_gen(qty,28,1))
        self.toolsmenu.add_command(label = 'Generate 31 Keys ^40 Ordered', command=lambda qty=31: self.codekey_gen(qty,40,1))
        self.menubar.add_cascade(label = 'Tools', menu = self.toolsmenu)
        self.config(menu = self.menubar)

        main_prog = ttk.Frame(self)
        top_area = ttk.Frame(main_prog)
        style1 = ttk.Style()
        self.tk.call("source", "azure.tcl")
        self.tk.call("set_theme", settings[0].lower())
        bold_font = tkFont.Font(size="10", weight="bold")

        # Main top entry items
        lbl1 = Label(top_area, text='Target (To):', font=('Helvetica 12'))
        lbl1.grid(row=0, column=0, sticky='E', padx=10, pady=(10,0))

        tebtn1 = ttk.Frame(top_area)
        tebtn1.grid(row=0, column=1, sticky='W', padx=10, pady=(10,5))
        self.target_station = ttk.Entry(tebtn1, width=12, font=('Helvetica 12'))
        self.target_station.grid(row=0, column=0, sticky='W', padx=0, pady=0)
        tobtn = ttk.Button(tebtn1, text='<- ID', command=self.fill_to)
        tobtn.grid(row=0,column=1, sticky='W', padx=(10,0))
        swapbtn = ttk.Button(tebtn1, text='Swap', command=self.swap_tofrom)
        swapbtn.grid(row=0,column=2, padx=(10,0))

        lbl2 = Label(top_area, text='Source (From):', font=('Helvetica 12'))
        lbl2.grid(row=1, column=0, sticky='E', padx=10, pady=(0,0))

        tebtn2 = ttk.Frame(top_area)
        tebtn2.grid(row=1, column=1, sticky='W', padx=10, pady=(0,10))
        self.source_station = ttk.Entry(tebtn2, width=12, font=('Helvetica 12'))
        self.source_station.grid(row=0, column=0, sticky='W', padx=0, pady=0)
        frombtn = ttk.Button(tebtn2, text='<- ID', command=self.fill_from)
        frombtn.grid(row=0,column=1, sticky='W', padx=(10,0))

        lbl3 = Label(top_area, text='Message:', font=('Helvetica 12'))
        lbl3.grid(row=2, column=0, sticky='E', padx=10, pady=(0,0))

        message_frame = tk.Frame(top_area)
        message_frame.grid(row=2, column=1, sticky="NSEW", padx=10, pady=(0,5))

        self.message_plain = Text(message_frame, wrap="none", width=32, font=('Helvetica 12'))
        scrollbar_y = ttk.Scrollbar(message_frame, command=self.message_plain.yview)
        scrollbar_x = ttk.Scrollbar(message_frame, orient="horizontal", command=self.message_plain.xview)
        self.message_plain.configure(yscrollcommand=scrollbar_y.set, xscrollcommand=scrollbar_x.set)
        scrollbar_y.grid(row=0, column=1, sticky="NS")
        scrollbar_x.grid(row=1, column=0, sticky="EW")
        self.message_plain.grid(row=0, column=0, sticky="NSEW")
        self.message_plain.bind('<Control-a>', self.select_all)

        message_frame.grid_rowconfigure(0, weight=1)
        message_frame.grid_columnconfigure(0, weight=1)
        top_area.grid_rowconfigure(2, weight=1)
        top_area.grid_columnconfigure(1, weight=1)

        self.switch_slinput = ttk.Checkbutton(top_area, text='Single Line', style='Switch.TCheckbutton', command=self.sl_toggle)
        self.switch_slinput.grid(row=3, column=0, sticky='E', padx=10, pady=(0,0))
        self.message_sl = ttk.Entry(top_area,width=32, font=('Helvetica 12'))
        self.message_sl.grid(row=3, column=1, sticky='W', padx=10, pady=(0,5))
        self.message_sl.config(state='disabled')

        lbl4 = Label(top_area, text='Key:', font=('Helvetica 12'))
        lbl4.grid(row=4, column=0, sticky='E', padx=10, pady=(0,10))
        self.codekeysel = ttk.Combobox(top_area, values="", state='readonly', width=32, font=('Helvetica 12'))
        self.codekeysel.grid(row=4, column =1 , sticky='W', padx=10, pady=(0,10))

        btn1 = ttk.Button(top_area, text='Reset', command=lambda ev=None: self.reset_all(), width=20)
        btn1.grid(row=5,column=0, columnspan=2)

        sep = ttk.Separator(top_area,orient='horizontal')
        sep.grid(row=6,column=0, columnspan=2, sticky='EW', pady=(10,10))

        # Tabs
        self.tabs = ttk.Notebook(main_prog)

        # right-click action
        self.rcmenu = Menu(self.tabs, tearoff = 0)
        self.rcmenu.add_command(label = 'Copy')

        # Sign tab
        sign_tab = ttk.Frame(self.tabs)
        self.tabs.add(sign_tab, text='Sign Message')

        self.add_dc = ttk.Checkbutton(sign_tab, text="Add Datecode")
        self.add_dc.grid(column=0, row=0, columnspan=2)
        self.add_dc.state(['!alternate'])

        btn1 = ttk.Button(sign_tab, text='Sign', command=self.sign_message, width=20)
        btn1.grid(row=1,column=0, columnspan=2,pady=10)

        self.sign_output = Text(sign_tab, width=48, height=8, padx=10, pady=10)
        self.sign_output.grid(row=2,column=0, sticky="NESW", padx=(20,0), pady=(0,10))
        sign_output_sb = ttk.Scrollbar(sign_tab, orient=tk.VERTICAL, command=self.sign_output.yview)
        self.sign_output.configure(yscroll=sign_output_sb.set)
        sign_output_sb.grid(row=2,column=1, sticky="NSEW", pady=(0,10))
        self.sign_output.bind('<Button-3>', lambda ev: self.copy_signed(ev))
        self.sign_output.tag_configure("bold_text", font=bold_font)
        sign_tab.grid_rowconfigure(0, weight=1)
        sign_tab.grid_columnconfigure(0, weight=1)

        # Validate Tab
        validate_tab = ttk.Frame(self.tabs)
        self.tabs.add(validate_tab, text='Validate Message')

        lbl5 = Label(validate_tab, text='CRC:')
        lbl5.grid(row=0, column=0, sticky='E', padx=10, pady=(10,0))
        self.crc_check = ttk.Entry(validate_tab, width=6, font=('Helvetica 12'))
        self.crc_check.grid(row=0, column=1, sticky='W', padx=10, pady=(10,0))

        btn2 = ttk.Button(validate_tab, text='Validate', command=self.validate_message, width=20)
        btn2.grid(row=1,column=0, columnspan=3, pady=10)

        self.validate_output = Text(validate_tab, width=48, height=8, padx=10, pady=10)
        self.validate_output.grid(row=2,column=0, columnspan=2, sticky="NESW", padx=(20,0), pady=(0,10))
        validate_output_sb = ttk.Scrollbar(validate_tab, orient=tk.VERTICAL, command=self.validate_output.yview)
        self.validate_output.configure(yscroll=validate_output_sb.set)
        validate_output_sb.grid(row=2,column=2, sticky="NESW", pady=(0,10))

        validate_tab.grid_rowconfigure(0, weight=1)
        validate_tab.grid_columnconfigure(0, weight=1)
        validate_tab.grid_columnconfigure(1, weight=1)

        self.validate_output.bind('<Button-3>', lambda ev: self.copy_validated(ev))
        self.validate_output.tag_configure("pass_font", foreground="#00A300", font=bold_font)
        self.validate_output.tag_configure("fail_font", foreground="red", font=bold_font)

        # layout
        top_area.grid(row=0,column=0, sticky="EW")
        self.tabs.grid(row=1,column=0, sticky="EW", padx=10, pady=10)

        main_prog.grid(row=0, column=0, sticky="NESW")
        main_prog.grid_rowconfigure(0, weight=1)
        main_prog.grid_columnconfigure(0, weight=1)

        self.grid_rowconfigure(0, weight=1)
        self.grid_columnconfigure(0, weight=1)

        self.tabs.bind("<<NotebookTabChanged>>", self.switch_to_tab)
        self.get_settings()

    def select_all(self, event):
        event.widget.tag_add('sel', '1.0', 'end')
        return 'break'

    def fill_to(self):
        self.target_station.delete(0,END)
        self.target_station.insert(END,settings[1].upper())

    def fill_from(self):
        self.source_station.delete(0,END)
        self.source_station.insert(END,settings[1].upper())

    def swap_tofrom(self):
            sendto = self.target_station.get()
            sendfrom = self.source_station.get()
            self.target_station.delete(0,END)
            self.source_station.delete(0,END)
            self.source_station.insert(END,sendto)
            self.target_station.insert(END,sendfrom)

    ## switch between entering each line, or enter one line
    def sl_toggle(self):

        if 'selected' in self.switch_slinput.state():
            # switch to alternate input selected
            sendto = self.target_station.get()
            sendfrom = self.source_station.get()
            message_plain = self.message_plain.get('1.0','end')

            message_plain=re.sub(r'[\x00-\x1f]', ' ', message_plain)
            message_plain=re.sub('\s{2,}', ' ', message_plain.strip())

            self.target_station.delete(0,END)
            self.source_station.delete(0,END)
            self.message_plain.delete('1.0',END)

            self.target_station.config(state='disabled')
            self.source_station.config(state='disabled')
            self.message_plain.config(state='disabled')
            self.message_sl.config(state='normal')

            if sendfrom!="" and sendto!="" and message_plain!="":
                self.message_sl.insert(END,sendfrom+": "+sendto+" "+message_plain)

        else:
            # switch to regular field-based input selected
            message_sl = self.message_sl.get()

            message_sl=re.sub(r'[\x00-\x1f]', ' ', message_sl)
            message_sl=re.sub('\s{2,}', ' ', message_sl.strip())

            self.target_station.config(state='normal')
            self.source_station.config(state='normal')
            self.message_plain.config(state='normal')
            self.message_sl.delete(0,END)
            self.message_sl.config(state='disabled')

            # match groups: 1=from, 2=to, 3=message
            find_msgitems = re.search(r"(.*?):\s+?(.*?)\s+?(.*)",message_sl)
            if find_msgitems:
                self.source_station.insert(END,find_msgitems[1])
                self.target_station.insert(END,find_msgitems[2])
                self.message_plain.insert(END,find_msgitems[3])

    ## reset all inputs
    def reset_all(self):
        self.target_station.delete(0,END)
        self.source_station.delete(0,END)
        self.message_plain.delete('1.0',END)
        self.message_sl.delete(0,END)
        self.crc_check.delete(0,END)
        self.sign_output.configure(state='normal')
        self.sign_output.delete("1.0", "end")
        self.sign_output.configure(state='disabled')
        self.validate_output.configure(state='normal')
        self.validate_output.delete("1.0", "end")
        self.validate_output.configure(state='disabled')

    ## switch between sign and validate stub
    def switch_to_tab(self, ev):
        pass

    ## Primary activity: Sign
    def sign_message(self):
        # determine whether to get entries or alternate input method
        if 'selected' in self.switch_slinput.state():
            # single line input selected
            message_sl = self.message_sl.get().upper()
            message_sl=re.sub(r'[\x00-\x1f]', ' ', message_sl)
            message_sl = re.sub('\s{2,}', ' ', message_sl)
            find_msgitems = re.search(r"(.*?):\s+?(.*?)\s+?(.*)",message_sl)
            if find_msgitems:
                sendto = find_msgitems[2]
                sendfrom = find_msgitems[1]
                sendmsg = find_msgitems[3]
            else:
                sendto=""
                sendfrom=""
                sendmsg=""
        else:
            # normal message input selected
            sendto = self.target_station.get().upper()
            sendfrom = self.source_station.get().upper()
            sendmsg = self.message_plain.get('1.0','end').upper()

        codekey = self.codekeysel.get()
        add_dc = self.add_dc.state()

        # pre-process
        sendto=sendto.strip()
        sendfrom=sendfrom.strip()
        sendmsg=re.sub(r'[\x00-\x1f]', ' ', sendmsg)
        sendmsg=re.sub('\s{2,}', ' ', sendmsg.strip())

        if 'selected' in add_dc:
            ecst = " "+self.encode_shorttime()
        else:
            ecst = ""

        self.sign_output.configure(state='normal')
        self.sign_output.delete("1.0", "end")

        if sendto=="" or sendfrom=="" or sendmsg=="" or codekey=="":
            messagebox.showwarning("Incomplete data","You must fill in all the information properly to sign a message.")
            return

        msgfmt = sendfrom+": "+sendto+" "+sendmsg+ecst
        newcrc = checksum(msgfmt,codekey)

        if newcrc==-1:
            messagebox.showwarning("CRC Error","Please check your key format and try again.")
            return

        if 'selected' in add_dc:
            self.sign_output.insert(END, "Datecode + CRC Signature:\n", "bold_text")
            self.sign_output.insert(END, ecst.strip()+" "+newcrc)
        else:
            self.sign_output.insert(END, "CRC Signature:\n", "bold_text")
            self.sign_output.insert(END, newcrc)

        self.sign_output.insert(END, "\nMessage Text to Validate:\n", "bold_text")
        self.sign_output.insert(END, sendmsg+ecst)
        self.sign_output.insert(END, "\nSource + Target + Message + CRC:\n", "bold_text")
        self.sign_output.insert(END, msgfmt+" "+newcrc)
        self.sign_output.configure(state='disabled')

    def copy_signed(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.sign_output.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.sign_output.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    ## Primary actvity: validate
    def validate_message(self):
        # determine to get entries or alternate input method
        if 'selected' in self.switch_slinput.state():
            message_sl = self.message_sl.get().upper()
            message_sl = re.sub('\s{2,}', ' ', message_sl)
            find_msgitems = re.search(r"(.*?):\s+?(.*?)\s+?(.*)",message_sl)
            if find_msgitems:
                sendto = find_msgitems[2]
                sendfrom = find_msgitems[1]
                sendmsg = find_msgitems[3]
            else:
                sendto=""
                sendfrom=""
                sendmsg=""
        else:
            sendto = self.target_station.get().upper()
            sendfrom = self.source_station.get().upper()
            sendmsg = self.message_plain.get('1.0','end').upper()

        codekey = self.codekeysel.get()
        sentcrc = self.crc_check.get().upper()

        # pre-process
        sendto=sendto.strip()
        sendfrom=sendfrom.strip()
        sendmsg=re.sub(r'[\x00-\x1f]', ' ', sendmsg)
        sendmsg=re.sub('\s{2,}', ' ', sendmsg.strip())

        message_full = sendfrom+": "+sendto+" "+sendmsg

        self.validate_output.configure(state='normal')
        self.validate_output.delete("1.0", "end")

        if sendto=="" or sendfrom=="" or sendmsg=="" or codekey=="":
            messagebox.showwarning("Incomplete data","You must fill in all the information properly to validate a message.")
            return

        checkcrc = checksum(message_full,codekey)
        if checkcrc==-1:
            messagebox.showwarning("Key Failed","Please check your key format and try again.")
            return

        # detect datecode at the end of the message
        tdc = ""
        find_dc = re.search(r"(.*?)(\#[A-Z0-9]+)",message_full)
        if find_dc:
            tdc = self.decode_shorttime(find_dc[2])

        if checkcrc == sentcrc:
            self.validate_output.insert(END,"PASSED: "+checkcrc, ("pass_font",))
        else:
            self.validate_output.insert(END,"FAILED: "+checkcrc, ("fail_font",))

        if tdc!="":
            self.validate_output.insert(END,"\n\nDatecode Detected: "+str(tdc) )
        self.validate_output.configure(state='disabled')

    def copy_validated(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.validate_output.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.validate_output.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    ## generate and display 24 random codekeys
    def codekey_gen(self, qty, segs, ordered):
        self.top = Toplevel(self)
        self.top.title("Generate Keys")
        self.top.geometry('500x450')
        self.top.resizable(width=False, height=False)

        mark = ttk.Label(self.top, text="Randomly Generated Keys")
        mark.grid(row=0,column=0)

        # text window
        self.top.codes = Text(self.top, wrap=WORD, width=60, height=24, font='TkFixedFont')
        self.top.codes.grid(row=1,column=0,padx=(8,8))
        self.top.codes.bind('<Button-3>', lambda ev: self.copy_codekey(ev))

        characters = string.ascii_uppercase + string.digits
        new_codekeys=""
        for i in range(0,qty):
            new_codekey = ''.join(random.choice(characters) for _ in range(segs))+"\n"
            if ordered==1:
                new_codekey=str(i+1)+": "+new_codekey
            new_codekeys+=new_codekey

        if ordered==1:
            self.top.codes.insert(END,"These keys are ordered and labeled. Copy a full line, including the label, to use as a labeled key.\n\n")
        self.top.codes.insert(END, str(new_codekeys)+"\n")

        self.top.focus()
        self.top.wait_visibility()
        self.top.grab_set()
        self.top.bind('<Escape>', lambda x: self.top.destroy())

    def copy_codekey(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.top.codes.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.top.codes.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    def get_settings(self):
        global settings
        codekey_opts = []

        if len(settings)<3:
            settings=["light","ERROR","ERROR"]

        if settings[2]=="ERROR" or settings[2]=="":
            # multiple conditions can cause [2] to be ERROR
            messagebox.showwarning("Configuration Error","Your configuration file is incomplete. Please review the documentation.")

        codekey_opts.append(settings[2])

        if len(settings)>3:
            for i in range(3, len(settings)):
                if settings[i]!="":
                    codekey_opts.append(settings[i])

        self.codekeysel['values'] = codekey_opts
        self.codekeysel.current(0)
        self.update()

    # imported from js8spotter v1.14b
    def decode_shorttime(self, ststamp):
        dcst=""
        # note that ststamp has # at position [0]. We can decode partial timestamps
        m = ""
        if len(ststamp) > 1:
            ma = ord(ststamp[1]) # Month, A-L (1-12)
            if ma>64 and ma<77: m = str(ma-64)
        else:
            m="0"

        d = ""
        if len(ststamp) > 2:
            da = ord(ststamp[2]) # Day, A-Z = 1-26, 0-4 = 27-31
            if da>47 and da<53: d = str((da-47)+26)
            if da>64 and da<91: d = str(da-64)
        else:
            d="0"

        h=""
        if len(ststamp) > 3:
            ha = ord(ststamp[3]) # Hour, A-W = 0-23
            if ha>64 and ha<88: h = str(ha-65)
        else:
            h="00"

        t=""
        if len(ststamp) > 4: # previous version had only three (plus #) characters, so we'll have this be optional
            ma = ord(ststamp[4]) # Minutes, 2min resolution, A-Z and 0-3 (A=0, B=2, C=4, etc)
            if ma>47 and ma<52: t = h+":"+str(((ma-48)+26)*2).zfill(2)
            if ma>64 and ma<91: t = h+":"+str((ma-65)*2).zfill(2)
        else:
            t = h+":00"

        if m != "" and d != "" and h !="" and t != "":
            dcst = str(m+"/"+d+" "+t)

        return dcst

    ## imported from js8spotter v1.14b -- Return the current time as an encode short time string
    def encode_shorttime(self):
        curtime=time.localtime(time.time())
        m=chr(curtime[1]+64)
        da=int(curtime[2])
        if da<27: d=chr(da+64)
        if da>26: d=chr((da-26)+47)
        h=chr(curtime[3]+65)
        mi=int(curtime[4]/2)+1+64
        if mi>90: mi-=43
        return "#"+m+d+h+chr(mi)

###

def checksum(message, codekey):
    checksum = ""

    out=hashlib.sha256((message+codekey).encode('utf-8')).hexdigest()
    decimal_value = int(hashlib.sha256(out.encode('utf-8')).hexdigest(),16)
    base36_chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

    while decimal_value > 0:
        checksum = base36_chars[decimal_value % 36] + checksum
        decimal_value //= 36

    return str(checksum)[-3:]

###

try:
    with open('msgauth.cfg') as f:
        settings = [line.rstrip('\n') for line in f]
    if settings[0].lower()!="light" and settings[0].lower()!="dark":
        settings[0]="light"
except FileNotFoundError:
    settings = ["light","ERROR","ERROR"]

if __name__ == "__main__":
    app = App()
    app.mainloop()

