#!/usr/bin/env python3
# Message Authenticator v0.7_021125. Visit https://kf7mix.com/ for information
# Special thanks to KD0QYN, KN4AM, and everyone else who contributed (see changelog)
#
# MIT License, Copyright 2025 Joseph D Lyman KF7MIX --- Permission is hereby granted,  free of charge, to any person obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:  The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.  The Software IS PROVIDED "AS IS",  WITHOUT WARRANTY OF ANY KIND,  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES  OF  MERCHANTABILITY,  FITNESS OR A PARTICULAR PURPOSE AND  NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR  COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,  DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import tkinter as tk
from tkinter import *
from tkinter import ttk, messagebox
import hashlib
import time
import datetime
import re
import random
import string

swname = "Message Authenticator"
fromtext = "de KF7MIX"
swversion = "0.70"
progsize = "500x480"
settings = []

class App(tk.Tk):
    def __init__(self):
        super().__init__()

        self.title(swname+" "+fromtext+" (v"+swversion+")")
        self.geometry(progsize)
        self.resizable(width=False, height=False)

        # Init the gui

        # Menu
        self.menubar = Menu(self)
        self.toolsmenu = Menu(self.menubar, tearoff = 0)
        self.toolsmenu.add_command(label = 'Generate 24 Keys ^25', command=lambda qty=24: self.codekey_gen(qty,25,0))
        self.toolsmenu.add_command(label = 'Generate 24 Keys ^36', command=lambda qty=24: self.codekey_gen(qty,36,0))
        self.toolsmenu.add_command(label = 'Generate 12 Keys ^28 Ordered', command=lambda qty=12: self.codekey_gen(qty,28,1))
        self.toolsmenu.add_command(label = 'Generate 24 Keys ^40 Ordered', command=lambda qty=24: self.codekey_gen(qty,40,1))
        self.menubar.add_cascade(label = 'Tools', menu = self.toolsmenu)
        self.config(menu = self.menubar)

        main_prog = ttk.Frame(self)
        top_area = ttk.Frame(main_prog)
        style1 = ttk.Style()
        self.tk.call("source", "azure.tcl")
        self.tk.call("set_theme", settings[0].lower())

        # Main top entry items
        lbl1 = Label(top_area, text='Target Station:', font=('Helvetica 12'))
        lbl1.grid(row=0, column=0, sticky='E', padx=10, pady=(10,0))
        self.target_station = ttk.Entry(top_area, width=12, font=('Helvetica 12'))
        self.target_station.grid(row=0, column=1, sticky='W', padx=10, pady=(10,5))

        lbl2 = Label(top_area, text='Source Station:', font=('Helvetica 12'))
        lbl2.grid(row=1, column=0, sticky='E', padx=10, pady=(0,0))
        self.source_station = ttk.Entry(top_area, width=12, font=('Helvetica 12'))
        self.source_station.grid(row=1, column=1, sticky='W', padx=10, pady=(0,5))

        lbl3 = Label(top_area, text='Message:', font=('Helvetica 12'))
        lbl3.grid(row=2, column=0, sticky='E', padx=10, pady=(0,0))
        self.message_plain = ttk.Entry(top_area,width=32, font=('Helvetica 12'))
        self.message_plain.grid(row=2, column=1, sticky='W', padx=10, pady=(0,5))

        lbl4 = Label(top_area, text='Key:', font=('Helvetica 12'))
        lbl4.grid(row=3, column=0, sticky='E', padx=10, pady=(0,10))
        self.codekeysel = ttk.Combobox(top_area, values="", state='readonly', width=32, font=('Helvetica 12'))
        self.codekeysel.grid(row=3, column =1 , sticky='W', padx=10, pady=(0,10))

        btn1 = tk.Button(top_area, text='Reset', command=lambda ev=None: self.switch_to_tab(ev), width=20)
        btn1.grid(row=4,column=0, columnspan=2)

        sep = ttk.Separator(top_area,orient='horizontal')
        sep.grid(row=5,column=0, columnspan=2, sticky='EW', pady=(10,10))

        # Tabs
        self.tabs = ttk.Notebook(main_prog, width=400)

        # right-click action
        self.rcmenu = Menu(self.tabs, tearoff = 0)
        self.rcmenu.add_command(label = 'Copy')

        # Sign tab
        sign_tab = ttk.Frame(self.tabs)
        self.tabs.add(sign_tab, text='Sign Message')

        self.add_dc = ttk.Checkbutton(sign_tab, text="Add Datecode")
        self.add_dc.grid(column=0, row=0, columnspan=2)
        self.add_dc.state(['!alternate'])

        btn1 = tk.Button(sign_tab, text='Sign', command=self.sign_message, width=20)
        btn1.grid(row=1,column=0, columnspan=2,pady=10)

        self.sign_output = Text(sign_tab, width=48, height=6, padx=10, pady=10)
        self.sign_output.grid(row=2,column=0, sticky="NESW", padx=(20,0), pady=(0,10))
        sign_output_sb = ttk.Scrollbar(sign_tab, orient=tk.VERTICAL, command=self.sign_output.yview)
        self.sign_output.configure(yscroll=sign_output_sb.set)
        sign_output_sb.grid(row=2,column=1, sticky="NS", pady=(0,10))
        self.sign_output.bind('<Button-3>', lambda ev: self.copy_signed(ev))

        # Validate Tab
        validate_tab = ttk.Frame(self.tabs)
        self.tabs.add(validate_tab, text='Validate Message')

        lbl5 = Label(validate_tab, text='CRC:')
        lbl5.grid(row=0, column=0, sticky='E', padx=10, pady=(10,0))
        self.crc_check = tk.Entry(validate_tab, width=6)
        self.crc_check.grid(row=0, column=1, sticky='W', padx=10, pady=(10,0))

        btn2 = tk.Button(validate_tab, text='Validate', command=self.validate_message, width=20)
        btn2.grid(row=1,column=0, columnspan=3, pady=10)

        self.validate_output = Text(validate_tab, width=48, height=6, padx=10, pady=10)
        self.validate_output.grid(row=2,column=0, columnspan=2, sticky="NESW", padx=(20,0), pady=(0,10))
        validate_output_sb = ttk.Scrollbar(validate_tab, orient=tk.VERTICAL, command=self.validate_output.yview)
        self.validate_output.configure(yscroll=validate_output_sb.set)
        validate_output_sb.grid(row=2,column=2, sticky="NS", pady=(0,10))
        self.validate_output.bind('<Button-3>', lambda ev: self.copy_validated(ev))

        # layout
        top_area.grid(row=0,column=0)
        self.tabs.grid(row=1,column=0, pady=(10,20))

        main_prog.grid(row=0, column=0, sticky="NESW")
        main_prog.grid_rowconfigure(0, weight=1)
        main_prog.grid_columnconfigure(0, weight=1)

        self.grid_rowconfigure(0, weight=1)
        self.grid_columnconfigure(0, weight=1)

        self.tabs.bind("<<NotebookTabChanged>>", self.switch_to_tab)
        self.get_settings()

    def switch_to_tab(self, ev):
        self.target_station.delete(0,END)
        self.source_station.delete(0,END)
        self.message_plain.delete(0,END)
        self.crc_check.delete(0,END)

        curtab = self.tabs.tab(self.tabs.select(), "text")
        if curtab=="Sign Message":
            self.source_station.insert(tk.END,settings[1].upper())

        if curtab=="Validate Message":
            self.target_station.insert(tk.END,settings[1].upper())

        return

    def sign_message(self):
        sendto = self.target_station.get().upper()
        sendfrom = self.source_station.get().upper()
        sendmsg = self.message_plain.get().upper()
        codekey = self.codekeysel.get()
        add_dc = self.add_dc.state()

        if 'selected' in add_dc:
            ecst = " "+self.encode_shorttime()
        else:
            ecst = ""

        self.sign_output.configure(state='normal')
        self.sign_output.delete("1.0", "end")

        if sendto=="" or sendfrom=="" or sendmsg=="" or codekey=="":
            messagebox.showwarning("Incomplete data","You must fill in all the fields to sign a message.")
            return

        msgfmt = sendfrom+": "+sendto+" "+sendmsg+ecst
        newcrc = checksum(msgfmt,codekey)

        if newcrc==-1:
            messagebox.showwarning("CRC Error","Please check your key format and try again.")
            return

        self.sign_output.insert(tk.END, "CRC Signature:\n"+newcrc+"\n\nFormatted Message (auto-copied to clipboard):\n"+msgfmt+" "+newcrc)
        self.sign_output.configure(state='disabled')

        self.clipboard_clear()
        self.clipboard_append(msgfmt+" "+newcrc)
        return

    def copy_signed(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.sign_output.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.sign_output.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    def validate_message(self):
        sendto = self.target_station.get().upper()
        sendfrom = self.source_station.get().upper()
        sendmsg = self.message_plain.get().upper()
        codekey = self.codekeysel.get()
        sentcrc = self.crc_check.get().upper()
        message_full = sendfrom+": "+sendto+" "+sendmsg

        self.validate_output.configure(state='normal')
        self.validate_output.delete("1.0", "end")

        if sendto=="" or sendfrom=="" or sendmsg=="" or codekey=="":
            messagebox.showwarning("Incomplete data","You must fill in all the fields to validate a message.")
            return

        checkcrc = checksum(message_full,codekey)
        if checkcrc==-1:
            messagebox.showwarning("Key Failed","Please check your key format and try again.")
            return

        # detect datecode at the end of the message
        tdc = ""
        find_dc = re.search(r"(.*?)(\#[A-Z0-9]+)",message_full)
        if find_dc:
            tdc = self.decode_shorttime(find_dc[2])

        if checkcrc == sentcrc:
            self.validate_output.insert(tk.END,"PASSED: "+checkcrc )
        else:
            self.validate_output.insert(tk.END,"FAILED: "+checkcrc )
            # run additional checks (trim)
            checkcrc = checksum(message_full.strip(),codekey)
            if checkcrc == sentcrc:
                self.validate_output.insert(tk.END,"\n\n!!! PASSED with extra spaces removed: "+checkcrc )

        if tdc!="":
            self.validate_output.insert(tk.END,"\n\nDatecode Detected: "+str(tdc) )
        return

    def copy_validated(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.validate_output.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.validate_output.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    ## generate and display 24 random codekeys
    def codekey_gen(self, qty, segs, ordered):
        self.top = Toplevel(self)
        self.top.title("Generate Keys")
        self.top.geometry('500x450')
        self.top.resizable(width=False, height=False)

        mark = ttk.Label(self.top, text="Randomly Generated Keys")
        mark.grid(row=0,column=0)

        # text window
        self.top.codes = Text(self.top, wrap=WORD, width=60, height=24, font='TkFixedFont')
        self.top.codes.grid(row=1,column=0,padx=(8,8))
        self.top.codes.bind('<Button-3>', lambda ev: self.copy_codekey(ev))

        characters = string.ascii_uppercase + string.digits
        new_codekeys=""
        for i in range(0,qty):
            new_codekey = ''.join(random.choice(characters) for _ in range(segs))+"\n"
            if ordered==1:
                new_codekey=str(i+1)+new_codekey
            new_codekeys+=new_codekey

        self.top.codes.insert(tk.END, str(new_codekeys)+"\n")

        self.top.focus()
        self.top.wait_visibility()
        self.top.grab_set()
        self.top.bind('<Escape>', lambda x: self.top.destroy())

        return

    def copy_codekey(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.top.codes.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.top.codes.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    def get_settings(self):
        global settings
        codekey_opts = []

        if len(settings)<3:
            settings=["light","ERROR","ERROR"]

        if settings[2]=="ERROR" or settings[2]=="":
            # multiple conditions can cause [2] to be ERROR
            messagebox.showwarning("Configuration Error","Your configuration file is incomplete. Please review the documentation.")

        codekey_opts.append(settings[2])

        if len(settings)>3:
            for i in range(3, len(settings)):
                if settings[i]!="":
                    codekey_opts.append(settings[i])

        self.codekeysel['values'] = codekey_opts
        self.codekeysel.current(0)
        self.update()
        return

    # imported from js8spotter v1.14b
    def decode_shorttime(self, ststamp):
        dcst=""
        # note that ststamp has # at position [0]. We can decode partial timestamps
        m = ""
        if len(ststamp) > 1:
            ma = ord(ststamp[1]) # Month, A-L (1-12)
            if ma>64 and ma<77: m = str(ma-64)
        else:
            m="0"

        d = ""
        if len(ststamp) > 2:
            da = ord(ststamp[2]) # Day, A-Z = 1-26, 0-4 = 27-31
            if da>47 and da<53: d = str((da-47)+26)
            if da>64 and da<91: d = str(da-64)
        else:
            d="0"

        h=""
        if len(ststamp) > 3:
            ha = ord(ststamp[3]) # Hour, A-W = 0-23
            if ha>64 and ha<88: h = str(ha-65)
        else:
            h="00"

        t=""
        if len(ststamp) > 4: # previous version had only three (plus #) characters, so we'll have this be optional
            ma = ord(ststamp[4]) # Minutes, 2min resolution, A-Z and 0-3 (A=0, B=2, C=4, etc)
            if ma>47 and ma<52: t = h+":"+str(((ma-48)+26)*2).zfill(2)
            if ma>64 and ma<91: t = h+":"+str((ma-65)*2).zfill(2)
        else:
            t = h+":00"

        if m != "" and d != "" and h !="" and t != "":
            dcst = str(m+"/"+d+" "+t)

        return dcst

    ## imported from js8spotter v1.14b -- Return the current time as an encode short time string
    def encode_shorttime(self):
        curtime=time.localtime(time.time())
        m=chr(curtime[1]+64)
        da=int(curtime[2])
        if da<27: d=chr(da+64)
        if da>26: d=chr((da-26)+47)
        h=chr(curtime[3]+65)
        mi=int(curtime[4]/2)+1+64
        if mi>90: mi-=43
        return "#"+m+d+h+chr(mi)

###

def checksum(message, codekey):
    checksum = ""

    out=hashlib.sha256((message+codekey).encode('utf-8')).hexdigest()
    decimal_value = int(hashlib.sha256(out.encode('utf-8')).hexdigest(),16)
    base36_chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

    while decimal_value > 0:
        checksum = base36_chars[decimal_value % 36] + checksum
        decimal_value //= 36

    return str(checksum)[-3:]

###

try:
    with open('msgauth.cfg') as f:
        settings = [line.rstrip('\n') for line in f]
    if settings[0].lower()!="light" and settings[0].lower()!="dark":
        settings[0]="light"
except FileNotFoundError:
    settings = ["light","ERROR","ERROR"]

if __name__ == "__main__":
    app = App()
    app.mainloop()

