#!/usr/bin/env python3
# Message Authenticator v0.5_020725. Visit https://kf7mix.com/ for information
# Special thanks to KD0QYN, KN4AM, and everyone else who contributed
#
# MIT License, Copyright 2025 Joseph D Lyman KF7MIX --- Permission is hereby granted,  free of charge, to any person obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:  The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.  The Software IS PROVIDED "AS IS",  WITHOUT WARRANTY OF ANY KIND,  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES  OF  MERCHANTABILITY,  FITNESS OR A PARTICULAR PURPOSE AND  NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR  COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,  DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import tkinter as tk
from tkinter import *
from tkinter import ttk, messagebox
import hashlib

progname = "Message Authenticator v0.5"
progsize = "500x450"
settings = []

class Window(Frame):
    def __init__(self, master=None):
        Frame.__init__(self, master)
        self.master = master

        # Init the gui
        main_prog = ttk.Frame(root)
        top_area = ttk.Frame(main_prog)
        style1 = ttk.Style()
        self.tk.call("source", "azure.tcl")
        self.tk.call("set_theme", settings[0].lower())

        # Main top entry items

        lbl1 = Label(top_area, text='Target Station:', font=('Helvetica 12'))
        lbl1.grid(row=0, column=0, sticky='E', padx=10, pady=(10,0))
        self.target_station = ttk.Entry(top_area, width=12, font=('Helvetica 12'))
        self.target_station.grid(row=0, column=1, sticky='W', padx=10, pady=(10,5))

        lbl2 = Label(top_area, text='Source Station:', font=('Helvetica 12'))
        lbl2.grid(row=1, column=0, sticky='E', padx=10, pady=(0,0))
        self.source_station = ttk.Entry(top_area, width=12, font=('Helvetica 12'))
        self.source_station.grid(row=1, column=1, sticky='W', padx=10, pady=(0,5))

        lbl3 = Label(top_area, text='Message:', font=('Helvetica 12'))
        lbl3.grid(row=2, column=0, sticky='E', padx=10, pady=(0,0))
        self.message_plain = ttk.Entry(top_area,width=32, font=('Helvetica 12'))
        self.message_plain.grid(row=2, column=1, sticky='W', padx=10, pady=(0,5))

        lbl4 = Label(top_area, text='Codeplug:', font=('Helvetica 12'))
        lbl4.grid(row=3, column=0, sticky='E', padx=10, pady=(0,10))
        self.codeplugsel = ttk.Combobox(top_area, values="", state='readonly', width=32, font=('Helvetica 12'))
        self.codeplugsel.grid(row=3, column =1 , sticky='W', padx=10, pady=(0,10))

        # Tabs
        tabs = ttk.Notebook(main_prog, width=400)

        # right-click action
        self.rcmenu = Menu(tabs, tearoff = 0)
        self.rcmenu.add_command(label = 'Copy')

        # Sign tab
        sign_tab = ttk.Frame(tabs)
        tabs.add(sign_tab, text='Sign Message')

        btn1 = tk.Button(sign_tab, text='Sign', command=self.sign_message, width=20)
        btn1.grid(row=0,column=0, columnspan=2,pady=10)

        self.sign_output = Text(sign_tab, width=48, height=6, padx=10, pady=10)
        self.sign_output.grid(row=1,column=0, sticky="NESW", padx=(20,0), pady=(0,10))
        sign_output_sb = ttk.Scrollbar(sign_tab, orient=tk.VERTICAL, command=self.sign_output.yview)
        self.sign_output.configure(yscroll=sign_output_sb.set)
        sign_output_sb.grid(row=1,column=1, sticky="NS", pady=(0,10))
        self.sign_output.bind('<Button-3>', lambda ev: self.copy_signed(ev))

        # Validate Tab
        validate_tab = ttk.Frame(tabs)
        tabs.add(validate_tab, text='Validate Message')

        lbl5 = Label(validate_tab, text='CRC:')
        lbl5.grid(row=0, column=0, sticky='E', padx=10, pady=(10,0))
        self.crc_check = tk.Entry(validate_tab, width=6)
        self.crc_check.grid(row=0, column=1, sticky='W', padx=10, pady=(10,0))

        btn2 = tk.Button(validate_tab, text='Validate', command=self.validate_message, width=20)
        btn2.grid(row=1,column=0, columnspan=3, pady=10)

        self.validate_output = Text(validate_tab, width=48, height=6, padx=10, pady=10)
        self.validate_output.grid(row=2,column=0, columnspan=2, sticky="NESW", padx=(20,0), pady=(0,10))
        validate_output_sb = ttk.Scrollbar(validate_tab, orient=tk.VERTICAL, command=self.validate_output.yview)
        self.validate_output.configure(yscroll=validate_output_sb.set)
        validate_output_sb.grid(row=2,column=2, sticky="NS", pady=(0,10))
        self.validate_output.bind('<Button-3>', lambda ev: self.copy_validated(ev))

        # layout
        top_area.grid(row=0,column=0)
        tabs.grid(row=1,column=0, pady=(10,20))

        main_prog.grid(row=0, column=0, sticky="NESW")
        main_prog.grid_rowconfigure(0, weight=1)
        main_prog.grid_columnconfigure(0, weight=1)

        root.grid_rowconfigure(0, weight=1)
        root.grid_columnconfigure(0, weight=1)

        self.get_settings()

        if settings[1]!="":
            self.source_station.insert(0,settings[1])

    def sign_message(self):
        sendto = self.target_station.get().upper()
        sendfrom = self.source_station.get().upper()
        sendmsg = self.message_plain.get().upper()
        plug = self.codeplugsel.get()

        self.sign_output.configure(state='normal')
        self.sign_output.delete("1.0", "end")

        if sendto=="" or sendfrom=="" or sendmsg=="" or plug=="":
            messagebox.showwarning("Incomplete data","You must fill in all the fields to sign a message.")
            return

        msgfmt = sendfrom+": "+sendto+" "+sendmsg
        newcrc = checksum(msgfmt,plug)

        if newcrc==-1:
            messagebox.showwarning("CRC Error","Please check your code plug format and try again.")
            return

        self.sign_output.insert(tk.END, "CRC Signature:\n"+newcrc+"\n\nFormatted Message (auto-copied to clipboard):\n"+msgfmt+" "+newcrc)
        self.sign_output.configure(state='disabled')

        self.clipboard_clear()
        self.clipboard_append(msgfmt+" "+newcrc)
        return

    def copy_signed(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.sign_output.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.sign_output.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    def validate_message(self):
        sendto = self.target_station.get().upper()
        sendfrom = self.source_station.get().upper()
        sendmsg = self.message_plain.get().upper()
        plug = self.codeplugsel.get()
        sentcrc = self.crc_check.get().upper()
        message_full = sendfrom+": "+sendto+" "+sendmsg

        self.validate_output.configure(state='normal')
        self.validate_output.delete("1.0", "end")

        if sendto=="" or sendfrom=="" or sendmsg=="" or plug=="":
            messagebox.showwarning("Incomplete data","You must fill in all the fields to validate a message.")
            return

        checkcrc = checksum(message_full,plug)
        if checkcrc==-1:
            messagebox.showwarning("Codeplug Failed","Please check your code plug format and try again.")
            return

        if checkcrc == sentcrc:
            self.validate_output.insert(tk.END,"PASSED: "+checkcrc )
        else:
            self.validate_output.insert(tk.END,"FAILED: "+checkcrc )
        return

    def copy_validated(self, ev):
        self.rcmenu.tk_popup(ev.x_root,ev.y_root)
        if self.validate_output.tag_ranges("sel"):
            self.clipboard_clear()
            text = self.validate_output.get('sel.first', 'sel.last')
            self.clipboard_append(text)

    def get_settings(self):
        global settings
        codeplug_opts = []

        if len(settings)<3:
            settings=["light","ERROR","ERROR"]

        if settings[2]=="ERROR" or settings[2]=="":
            # multiple conditions can cause [2] to be ERROR
            messagebox.showwarning("Configuration Error","Your configuration file is incomplete. Please review the documentation.")

        codeplug_opts.append(settings[2])

        if len(settings)>3:
            for i in range(3, len(settings)):
                if settings[i]!="":
                    codeplug_opts.append(settings[i])

        self.codeplugsel['values'] = codeplug_opts
        self.codeplugsel.current(0)
        self.update()
        return

###

def checksum(message, plug):
    i=0
    out=""
    checksum = ""

    for char in message:
        pf = plug[i]
        try:
            pv = int(plug[i+1:i+3])
        except ValueError:
            return -1
        c = ord(char)
        if pf=="+": o = ((c+pv)-32) % (127-32) + 32
        if pf=="-": o = ((c-pv)-32) % (127-32) + 32
        i+=3
        if i>=len(plug): i=0
        out+=chr(o)

    decimal_value = int(hashlib.sha256(out.encode('ascii')).hexdigest(),16)
    base36_chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

    while decimal_value > 0:
        checksum = base36_chars[decimal_value % 36] + checksum
        decimal_value //= 36

    return str(checksum)[-3:]

try:
    with open('msgauth.cfg') as f:
        settings = [line.rstrip('\n') for line in f]
    if settings[0].lower()!="light" and settings[0].lower()!="dark":
        settings[0]="light"
except FileNotFoundError:
    settings = ["light","ERROR","ERROR"]

root = Tk()
app = Window(root)
root.wm_title(progname)
root.geometry(progsize)
root.resizable(width=False, height=False)
root.mainloop()


